<?php
namespace Samba\M2Integration\Model;

use DateTimeInterface;
use Magento\Framework\Stdlib\DateTime\DateTime;
use Samba\M2Integration\Model\ResourceModel\Queue as QueueResource;
use Samba\M2Integration\Model\ResourceModel\Queue\CollectionFactory;
use Samba\M2Integration\Model\QueueFactory;
use Magento\Framework\Serialize\Serializer\Json;

class QueueManager
{
    /**
     * @var QueueFactory
     */
    private $queueFactory;

    /**
     * @var QueueResource
     */
    private $queueResource;

    /**
     * @var CollectionFactory
     */
    private $collectionFactory;

    /**
     * @var Json
     */
    private $serializer;

    /**
     * @var DateTime
     */
    private $dateTime;

    public function __construct(
        QueueFactory $queueFactory,
        QueueResource $queueResource,
        CollectionFactory $collectionFactory,
        Json $serializer,
        DateTime $dateTime
    ) {
        $this->queueFactory = $queueFactory;
        $this->queueResource = $queueResource;
        $this->collectionFactory = $collectionFactory;
        $this->serializer = $serializer;
        $this->dateTime = $dateTime;
    }

    public function enqueue(string $type, DateTimeInterface $scheduledAt = null)
    {
        $scheduleDate = $scheduledAt ? $this->formatDate($scheduledAt) : $this->dateTime->gmtDate();

        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('type', $type);
        $collection->addFieldToFilter('status', QueueStatus::PENDING);
        $collection->setPageSize(1);
        $collection->setCurPage(1);
        $existing = $collection->getFirstItem();

        if ($existing && $existing->getId()) {
            return null;
        }

        $queue = $this->queueFactory->create();
        $queue->setData([
            'type' => $type,
            'status' => QueueStatus::PENDING,
            'scheduled_at' => $scheduleDate,
        ]);

        $this->queueResource->save($queue);

        return $queue;
    }

    /**
     * @return Queue[]
     */
    public function getPendingJobs(int $limit = 10): array
    {
        $collection = $this->collectionFactory->create();
        $collection->addFieldToFilter('status', QueueStatus::PENDING);
        $collection->addFieldToFilter('scheduled_at', ['lteq' => $this->dateTime->gmtDate()]);
        $collection->setOrder('scheduled_at', 'ASC');
        $collection->setPageSize($limit);

        return $collection->getItems();
    }

    public function markProcessing(Queue $queue): void
    {
        $queue->setData('status', QueueStatus::PROCESSING);
        $queue->setData('message', null);
        $this->queueResource->save($queue);
    }

    public function markComplete(Queue $queue): void
    {
        $queue->setData('status', QueueStatus::COMPLETE);
        $queue->setData('executed_at', $this->dateTime->gmtDate());
        $queue->setData('message', null);
        $this->queueResource->save($queue);
    }

    public function markFailed(Queue $queue, string $message): void
    {
        $queue->setData('status', QueueStatus::FAILED);
        $queue->setData('executed_at', $this->dateTime->gmtDate());
        $queue->setData('message', $message);
        $this->queueResource->save($queue);
    }

    private function formatDate(DateTimeInterface $dateTime): string
    {
        return $dateTime->format('Y-m-d H:i:s');
    }
}
