<?php

declare(strict_types=1);

namespace Samba\M2Integration\Model\FeedGenerator;

use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\Catalog\Model\ResourceModel\Product\Collection as ProductCollection;
use Magento\CatalogInventory\Api\StockRegistryInterface;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Store\Model\StoreManagerInterface;
use Samba\M2Integration\Helper\ExportHelper;
use Samba\M2Integration\Model\FeedGeneratorInterface;
use Samba\M2Integration\Model\FileWriter;

class Products implements FeedGeneratorInterface
{
    const PRODUCT_PRICE_BEFORE_DISCOUNT = 'samba_config/products/product_price_before_discount';
    const PRODUCT_IMAGE = 'samba_config/products/product_image';
    const PRODUCT_DESCRIPTION = 'samba_config/products/product_description';
    const PRODUCT_STOCK = 'samba_config/products/product_stock';
    const PRODUCT_CATEGORIES = 'samba_config/products/product_categories';
    const PRODUCT_VARIANTS = 'samba_config/products/product_variants';
    const BATCH_SIZE = 500;

    /** @var CollectionFactory */
    private $productCollectionFactory;

    /** @var StockRegistryInterface */
    private $stockRegistry;

    /** @var Configurable */
    private $configurableProduct;

    /** @var ExportHelper */
    private $exportHelper;

    /** @var ProductRepositoryInterface */
    private $productRepository;

    /** @var StoreManagerInterface */
    private $storeManager;

    /** @var CategoryRepositoryInterface */
    private $categoryRepository;

    /** @var FileWriter */
    private $fileWriter;

    /**
     * @param CollectionFactory $productCollectionFactory
     * @param StockRegistryInterface $stockRegistry
     * @param Configurable $configurableProduct
     * @param ExportHelper $exportHelper
     * @param ProductRepositoryInterface $productRepository
     * @param StoreManagerInterface $storeManager
     * @param CategoryRepositoryInterface $categoryRepository
     * @param FileWriter $fileWriter
     */
    public function __construct(
        CollectionFactory $productCollectionFactory,
        StockRegistryInterface $stockRegistry,
        Configurable $configurableProduct,
        ExportHelper $exportHelper,
        ProductRepositoryInterface $productRepository,
        StoreManagerInterface $storeManager,
        CategoryRepositoryInterface $categoryRepository,
        FileWriter $fileWriter
    ) {
        $this->productCollectionFactory = $productCollectionFactory;
        $this->stockRegistry = $stockRegistry;
        $this->configurableProduct = $configurableProduct;
        $this->exportHelper = $exportHelper;
        $this->productRepository = $productRepository;
        $this->storeManager = $storeManager;
        $this->categoryRepository = $categoryRepository;
        $this->fileWriter = $fileWriter;
    }


    public function generateFeed(): void
    {
        $storeId = $this->exportHelper->getStoreId();
        $stream = $this->fileWriter->openFile(ExportHelper::PRODUCTS_FILE);

        $xmlWriter = new \XMLWriter();
        $xmlWriter->openMemory();
        $xmlWriter->setIndent(true);
        $xmlWriter->startDocument('1.0', 'UTF-8');
        $xmlWriter->startElement('PRODUCTS');

        $loop = 1;
        do {
            $collection = $this->load($loop);
            $size = count($collection);

            foreach ($collection as $p) {
                if (($productTitle = $p->getName()) == null) {
                    $productTitle = '';
                }
                if (($productPrice = $p->getPrice()) == null) {
                    $productPrice = '0';
                }

                $xmlWriter->startElement('PRODUCT');
                $xmlWriter->writeElement('PRODUCT_ID', $p->getId());

                if ($productTitle != null) {
                    $xmlWriter->writeElement('TITLE', htmlspecialchars($productTitle));
                }

                if ((bool)$this->exportHelper->getConfig(self::PRODUCT_PRICE_BEFORE_DISCOUNT)) {
                    if (($productSpecialPrice = $this->getSpecialPriceById($p->getId())) != null) {
                        $xmlWriter->writeElement('PRICE', $productSpecialPrice);

                        if ($productPrice != null) {
                            $xmlWriter->writeElement('PRICE_BEFORE_DISCOUNT', $productPrice);
                        }
                    } else {
                        if ($productPrice != null) {
                            $xmlWriter->writeElement('PRICE', $productPrice);
                        }
                    }
                } else {
                    if ($productPrice != null) {
                        $xmlWriter->writeElement('PRICE', $productPrice);
                    }
                }

                $xmlWriter->writeElement('URL', htmlspecialchars($p->getProductUrl() ?? ''));

                if ((bool)$this->exportHelper->getConfig(self::PRODUCT_IMAGE)) {
                    if (($productImage = $this->getProductImage($p, $storeId)) != null) {
                        $xmlWriter->writeElement('IMAGE', $productImage);
                    }
                }

                if ((bool)$this->exportHelper->getConfig(self::PRODUCT_DESCRIPTION)) {
                    if ($p->getDescription() != null) {
                        $xmlWriter->writeElement('DESCRIPTION', htmlspecialchars(strip_tags($p->getDescription())));
                    }
                }

                if ((bool)$this->exportHelper->getConfig(self::PRODUCT_STOCK)) {
                    if (($productStock = $this->getStockStatus($p->getSku())) != null && $productStock > 0) {
                        $xmlWriter->writeElement('STOCK', (string)$this->getStockItem($p->getSku()));
                    }
                }

                if ((bool)$this->exportHelper->getConfig(self::PRODUCT_CATEGORIES)) {
                    if (($productCategories = $this->getProductCategories($p, $storeId)) != null) {
                        $xmlWriter->writeElement('CATEGORYTEXT', $productCategories);
                    }
                }

                if ((bool)$this->exportHelper->getConfig(self::PRODUCT_VARIANTS)) {
                    if ($p->getTypeId() == 'configurable') {
                        $configurable = $this->productRepository->getById($p->getId());
                        $array = $configurable->getTypeInstance()->getUsedProductCollection($configurable)->getData();

                        foreach ($array as $item) {
                            $simple = $this->productRepository->getById($item['entity_id']);

                            $xmlWriter->startElement('VARIANT');
                            $xmlWriter->writeElement('PRODUCT_ID', $simple->getId());
                            $xmlWriter->writeElement('TITLE', htmlspecialchars($simple->getName()));
                            $xmlWriter->writeElement(
                                'DESCRIPTION',
                                htmlspecialchars(strip_tags($simple->getDescription() ?? ''))
                            );
                            $xmlWriter->writeElement('PRICE', $simple->getPrice());

                            if (($vProductStock = $this->getStockStatus($simple->getSku())) != null
                                && $vProductStock > 0) {
                                $xmlWriter->writeElement('STOCK', (string)$this->getStockItem($simple->getSku()));
                            }

                            if (($productImage = $this->getProductImage($simple, $storeId)) != null) {
                                $xmlWriter->writeElement('IMAGE', $productImage);
                            }

                            $xmlWriter->writeElement('URL', htmlspecialchars($simple->getProductUrl() ?? ''));
                            $xmlWriter->endElement(); //variant
                        }
                    }
                }
                $xmlWriter->endElement(); //product
            }
            $stream->write((string)$xmlWriter->flush(true));
            $this->productRepository->cleanCache();
            $loop++;
        } while ($size === self::BATCH_SIZE);

        $xmlWriter->endElement(); //products

        $stream->write((string)$xmlWriter->flush(true));
        $this->fileWriter->closeFile($stream);
    }

    /**
     * @param int $page
     * @return ProductCollection
     */
    private function load(int $page): ProductCollection
    {
        $storeId = $this->exportHelper->getStoreId();

        $collection = $this->productCollectionFactory->create();
        $collection->addFieldToSelect('*')
            //->addAttributeToFilter('status', Status::STATUS_ENABLED)
            ->addStoreFilter($storeId);
        $collection->setPageSize(self::BATCH_SIZE);
        $collection->setCurPage($page);
        $collection->load();

        return $collection;
    }

    public function getStockItem($sku)
    {
        return $this->stockRegistry->getStockItemBySku($sku)->getQty();
    }

    public function getStockStatus($sku)
    {
        $stockItems = $this->getStockItem($sku);
        if ($stockItems > 0) {
            return true;
        } else {
            return false;
        }
    }

    protected function getProductImage($product, $storeId)
    {
        $store = $this->storeManager->getStore($storeId);
        return $store->getBaseUrl(
                \Magento\Framework\UrlInterface::URL_TYPE_MEDIA
            ) . 'catalog/product' . $product->getImage();
    }

    public function getSpecialPriceById($id)
    {
        $product = $this->productRepository->getById($id);
        return $product->getSpecialPrice();
    }

    protected function getCategoryNameById($id, $storeId = null)
    {
        $categoryInstance = $this->categoryRepository->get($id, $storeId);

        return $categoryInstance->getName();
    }

    protected function getProductCategories($product, $storeId)
    {
        $categories = [];
        foreach ($product->getCategoryIds() as $categoryId) {
            $categories[] = $this->getCategoryNameById($categoryId, $storeId);
        }

        if (count($categories) > 0) {
            return implode(' | ', $categories);
        } else {
            return null;
        }
    }
}
