<?php

declare(strict_types=1);

namespace Samba\M2Integration\Model\FeedGenerator;

use Magento\Sales\Model\ResourceModel\Order\Collection as OrderCollection;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;
use Samba\M2Integration\Helper\ExportHelper;
use Samba\M2Integration\Model\FeedGeneratorInterface;
use Samba\M2Integration\Model\FileWriter;

class Orders implements FeedGeneratorInterface
{
    const ORDER_CUSTOMER_ID = 'samba_config/orders/order_customer_id';
    const ORDER_STATUS = 'samba_config/orders/order_status';
    const ORDER_EMAIL = 'samba_config/orders/order_email';
    const ORDER_PHONE = 'samba_config/orders/order_phone';
    const ORDER_ZIP_CODE = 'samba_config/orders/order_zip_code';
    const ORDER_PERIOD = 'samba_config/orders/order_period';
    const BATCH_SIZE = 500;

    /** @var CollectionFactory */
    private $orderCollectionFactory;

    /** @var ExportHelper */
    private $exportHelper;

    /** @var FileWriter */
    private $fileWriter;

    /**
     * @param CollectionFactory $orderCollectionFactory
     * @param ExportHelper $exportHelper
     * @param FileWriter $fileWriter
     */
    public function __construct(
        CollectionFactory $orderCollectionFactory,
        ExportHelper $exportHelper,
        FileWriter $fileWriter
    ) {
        $this->orderCollectionFactory = $orderCollectionFactory;
        $this->exportHelper = $exportHelper;
        $this->fileWriter = $fileWriter;
    }


    public function generateFeed(): void
    {
        $stream = $this->fileWriter->openFile(ExportHelper::ORDERS_FILE);

        $xmlWriter = new \XMLWriter();
        $xmlWriter->openMemory();
        $xmlWriter->setIndent(true);
        $xmlWriter->startDocument('1.0', 'UTF-8');
        $xmlWriter->startElement('ORDERS');

        $loop = 1;
        do {
            $collection = $this->load($loop);
            $size = count($collection);
            foreach ($collection as $k => $o) {
                $xmlWriter->startElement('ORDER');
                $xmlWriter->writeElement('ORDER_ID', $o->getId());

                if ((bool)$this->exportHelper->getConfig(self::ORDER_CUSTOMER_ID)) {
                    if (($customerId = $o->getCustomerId()) != null) {
                        $xmlWriter->writeElement('CUSTOMER_ID', $customerId);
                    }
                }

                $xmlWriter->writeElement('CREATED_ON', $this->exportHelper->dateFormat($o->getCreatedAt()));

                if ((bool)$this->exportHelper->getConfig(self::ORDER_STATUS)) {
                    if (($orderStatus = $this->getOrderStatus($o->getStatus())) != null) {
                        $xmlWriter->writeElement('STATUS', $orderStatus);
                    }
                }

                if ((bool)$this->exportHelper->getConfig(self::ORDER_EMAIL)) {
                    $xmlWriter->writeElement('EMAIL', $o->getCustomerEmail());
                }

                if ((bool)$this->exportHelper->getConfig(self::ORDER_PHONE)) {
                    $xmlWriter->writeElement('PHONE', $o->getBillingAddress()->getTelephone());
                }

                if ((bool)$this->exportHelper->getConfig(self::ORDER_ZIP_CODE)) {
                    $xmlWriter->writeElement('ZIP_CODE', $o->getBillingAddress()->getPostcode());
                }


                $xmlWriter->startElement('ITEMS');

                foreach ($o->getItems() as $p) {
                    $xmlWriter->startElement('ITEM');
                    $xmlWriter->writeElement('PRODUCT_ID', $p->getProductId());
                    $xmlWriter->writeElement('AMOUNT', strval((int)$p->getQtyOrdered()));
                    $xmlWriter->writeElement('PRICE', (string)$p->getPrice());
                    $xmlWriter->endElement(); //ITEM
                }

                $xmlWriter->endElement(); //ITEMS
                $xmlWriter->endElement(); //order
            }

            $stream->write((string)$xmlWriter->flush(true));

            $loop++;
        } while ($size === self::BATCH_SIZE);

        $xmlWriter->endElement(); //orders

        $stream->write((string)$xmlWriter->flush(true));
        $this->fileWriter->closeFile($stream);
    }

    /**
     * @param string $status
     * @return string|null
     */
    private function getOrderStatus(string $status): ?string
    {
        switch ($status) {
            case 'canceled':
                return 'canceled';
            case 'complete':
                return 'finished';
            default:
                return 'created';
        }

        return null;
    }

    /**
     * @param int $page
     * @return OrderCollection
     */
    private function load(int $page): OrderCollection
    {
        $storeId = $this->exportHelper->getStoreId();
        $collection = $this->orderCollectionFactory->create();
        $collection->addAttributeToSelect("*")
            ->addAttributeToFilter("store_id", array("eq" => $storeId));
        $collection->setPageSize(self::BATCH_SIZE);
        $collection->setCurPage($page);

        if ((bool)$this->exportHelper->getConfig(self::ORDER_PERIOD)) {
            $collection->addFieldToFilter('created_at', ['gt' => new \Zend_Db_Expr('NOW() - INTERVAL 3 MONTH')]);
        }

        $collection->load();

        return $collection;
    }
}
