<?php

declare(strict_types=1);

namespace Samba\M2Integration\Model\FeedGenerator;

use Magento\Customer\Model\AddressFactory;
use Magento\Customer\Model\CustomerFactory;
use Magento\Customer\Model\ResourceModel\Customer\Collection as CustomerCollection;
use Magento\Customer\Model\ResourceModel\Customer\CollectionFactory;
use Magento\Newsletter\Model\Subscriber;
use Samba\M2Integration\Helper\ExportHelper;
use Samba\M2Integration\Model\FeedGeneratorInterface;
use Samba\M2Integration\Model\FileWriter;

class Customers implements FeedGeneratorInterface
{
    const CUSTOMER_FIST_NAME = 'samba_config/customers/customer_first_name';
    const CUSTOMER_LAST_NAME = 'samba_config/customers/customer_last_name';
    const CUSTOMER_EMAIL = 'samba_config/customers/customer_email';
    const CUSTOMER_PHONE = 'samba_config/customers/customer_phone';
    const CUSTOMER_ZIP_CODE = 'samba_config/customers/customer_zip_code';
    const CUSTOMER_REGISTRATION_DATE = 'samba_config/customers/customer_registration_date';
    const CUSTOMER_PERIOD = 'samba_config/customers/customer_period';
    const BATCH_SIZE = 500;

    /** @var CollectionFactory */
    private $customerCollectionFactory;

    /** @var Subscriber */
    private $subscriber;

    /** @var CustomerFactory */
    private $customerFactory;

    /** @var AddressFactory */
    private $addressFactory;

    /** @var ExportHelper */
    private $exportHelper;

    /** @var FileWriter */
    private $fileWriter;

    /**
     * @param CollectionFactory $customerCollectionFactory
     * @param Subscriber $subscriber
     * @param CustomerFactory $customerFactory
     * @param AddressFactory $addressFactory
     * @param ExportHelper $exportHelper
     * @param FileWriter $fileWriter
     */
    public function __construct(
        CollectionFactory $customerCollectionFactory,
        Subscriber $subscriber,
        CustomerFactory $customerFactory,
        AddressFactory $addressFactory,
        ExportHelper $exportHelper,
        FileWriter $fileWriter
    ) {
        $this->customerCollectionFactory = $customerCollectionFactory;
        $this->subscriber = $subscriber;
        $this->customerFactory = $customerFactory;
        $this->addressFactory = $addressFactory;
        $this->exportHelper = $exportHelper;
        $this->fileWriter = $fileWriter;
    }


    public function generateFeed(): void
    {
        $stream = $this->fileWriter->openFile(ExportHelper::CUSTOMERS_FILE);

        $xmlWriter = new \XMLWriter();
        $xmlWriter->openMemory();
        $xmlWriter->setIndent(true);
        $xmlWriter->startDocument('1.0', 'UTF-8');
        $xmlWriter->startElement('CUSTOMERS');

        $loop = 1;
        do {
            $collection = $this->load($loop);
            $size = count($collection);
            foreach ($collection as $k => $c) {
                $xmlWriter->startElement('CUSTOMER');
                $xmlWriter->writeElement('CUSTOMER_ID', $c->getId());

                $billingAddress = $this->getBillingAddress($c->getId());

                if ((bool)$this->exportHelper->getConfig(self::CUSTOMER_FIST_NAME)) {
                    if (($customerFirstName = $this->getFirstName($billingAddress)) != null) {
                        $xmlWriter->writeElement('FIRST_NAME', $customerFirstName);
                    }
                }

                if ((bool)$this->exportHelper->getConfig(self::CUSTOMER_LAST_NAME)) {
                    if (($customerLastName = $this->getLastName($billingAddress)) != null) {
                        $xmlWriter->writeElement('LAST_NAME', $customerLastName);
                    }
                }

                if ((bool)$this->exportHelper->getConfig(self::CUSTOMER_EMAIL)) {
                    $xmlWriter->writeElement('EMAIL', $c->getEmail());
                }

                if ((bool)$this->exportHelper->getConfig(self::CUSTOMER_PHONE)) {
                    if (($customerPhone = $this->getCustomerPhone($billingAddress)) != null) {
                        $xmlWriter->writeElement('PHONE', $customerPhone);
                    }
                }

                if ((bool)$this->exportHelper->getConfig(self::CUSTOMER_ZIP_CODE)) {
                    if (($customerZipCode = $this->getCustomerZipCode($billingAddress)) != null) {
                        $xmlWriter->writeElement('ZIP_CODE', $customerZipCode);
                    }
                }

                $xmlWriter->writeElement('NEWSLETTER_FREQUENCY', $this->getCustomerNewsletter($c->getId()));

                if ((bool)$this->exportHelper->getConfig(self::CUSTOMER_REGISTRATION_DATE)) {
                    $xmlWriter->writeElement('REGISTRATION', $this->exportHelper->dateFormat($c->getCreatedAt()));
                }

                $xmlWriter->endElement(); //customer
            }

            $stream->write((string)$xmlWriter->flush(true));

            $loop++;
        } while ($size === self::BATCH_SIZE);

        $xmlWriter->endElement(); //customers

        $stream->write((string)$xmlWriter->flush(true));
        $this->fileWriter->closeFile($stream);
    }

    /**
     * @param int $page
     * @return CustomerCollection
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function load(int $page): CustomerCollection
    {
        $storeId = $this->exportHelper->getStoreId();
        $collection = $this->customerCollectionFactory->create();
        $collection->addAttributeToSelect([
            'email',
            'created_at',
            'firstname',
            'lastname',
        ])
            ->addAttributeToFilter("store_id", array("eq" => $storeId));
        $collection->setPageSize(self::BATCH_SIZE);
        $collection->setCurPage($page);

        if ((bool)$this->exportHelper->getConfig(self::CUSTOMER_PERIOD)) {
            $collection->addFieldToFilter('created_at', ['gt' => new \Zend_Db_Expr('NOW() - INTERVAL 3 MONTH')]);
        }
        $collection->load();

        return $collection;
    }

    protected function getBillingAddress($customerId)
    {
        if ($customerId == null) {
            return null;
        }

        $customer = $this->customerFactory->create()->load($customerId);
        $billingAddressId = $customer->getDefaultBilling();
        if ($billingAddressId == null) {
            return null;
        }

        return $this->addressFactory->create()->load($billingAddressId);
    }

    protected function getFirstName($billingAddress)
    {
        if ($billingAddress != null) {
            return htmlspecialchars($billingAddress->getFirstname() ?? '');
        }

        return null;
    }

    protected function getLastName($billingAddress)
    {
        if ($billingAddress != null) {
            return htmlspecialchars($billingAddress->getLastname() ?? '');
        }

        return null;
    }

    protected function getCustomerPhone($billingAddress)
    {
        if ($billingAddress != null) {
            return htmlspecialchars($billingAddress->getTelephone() ?? '');
        }

        return null;
    }

    protected function getCustomerZipCode($billingAddress)
    {
        if ($billingAddress != null) {
            return htmlspecialchars($billingAddress->getPostcode() ?? '');
        }

        return null;
    }

    protected function getCustomerNewsletter($customerId)
    {
        $userIsSubscriber = $this->subscriber->loadByCustomerId($customerId);

        if ($userIsSubscriber->isSubscribed()) {
            return 'every day';
        } else {
            return 'never';
        }
    }
}
