<?php
declare(strict_types=1);

namespace Samba\M2Integration\Cron;

use Samba\M2Integration\Model\FeedGenerator\Products as ProductFeedGenerator;
use Samba\M2Integration\Model\FeedGenerator\Orders as OrderFeedGenerator;
use Samba\M2Integration\Model\FeedGenerator\Customers as CustomerFeedGenerator;
use Samba\M2Integration\Model\Queue;
use Samba\M2Integration\Model\QueueManager;
use Psr\Log\LoggerInterface;

class QueueProcessor
{
    /**
     * @var ProductFeedGenerator
     */
    private $productFeedGenerator;

    /**
     * @var OrderFeedGenerator
     */
    private $orderFeedGenerator;

    /**
     * @var CustomerFeedGenerator
     */
    private $customerFeedGenerator;

    /**
     * @var QueueManager
     */
    private $queueManager;

    /**
     * Constructor
     *
     * @param LoggerInterface $logger
     */
    public function __construct(
        ProductFeedGenerator $productFeedGenerator,
        OrderFeedGenerator $orderFeedGenerator,
        CustomerFeedGenerator $customerFeedGenerator,
        QueueManager $queueManager,
        private LoggerInterface $logger
    ) {
        $this->productFeedGenerator = $productFeedGenerator;
        $this->orderFeedGenerator = $orderFeedGenerator;
        $this->customerFeedGenerator = $customerFeedGenerator;
        $this->queueManager = $queueManager;
    }

    /**
     * Execute the cron
     *
     * @return void
     */
    public function execute(): void
    {
        $queues = $this->queueManager->getPendingJobs();

        if ($queues === []) {
            return;
        }

        $handlers = ['products', 'customers', 'orders'];

        foreach ($queues as $queue) {
            $type = (string) $queue->getData('type');

            // if (!isset($this->handlers[$type])) {
            if (in_array($type, $handlers) === false) {
                $this->queueManager->markFailed($queue, sprintf('No handler configured for type "%s"', $type));
                $this->log('warning', 'Samba scheduler missing handler', ['type' => $type, 'queue_id' => $queue->getId()]);
                continue;
            }

            $this->queueManager->markProcessing($queue);

            try {
                if ($type === 'products') {
                    $this->productFeedGenerator->generateFeed();
                } else if ($type === 'customers') {
                    $this->customerFeedGenerator->generateFeed();
                } else if ($type === 'orders') {
                    $this->orderFeedGenerator->generateFeed();
                } else {
                    $this->jobManager->markFailed($queue, sprintf('Handler for type "%s" is invalid', $type));
                    $this->log('error', 'Samba scheduler handler is not an object', ['type' => $type, 'queue_id' => $queue->getId()]);
                    continue;
                }

                $this->queueManager->markComplete($queue);
                $this->log('info', 'Samba scheduler job completed');
            } catch (\Throwable $exception) {
                $this->queueManager->markFailed($queue, $exception->getMessage());
                $this->log('error', 'Samba scheduler job failed');
            }
        }
    }

    private function log(string $level, string $message, array $context = []): void
    {
        if ($this->logger === null) {
            return;
        }

        $level = $this->normalizeLevel($level);
        $this->logger->log($level, $message, $context);
    }

    private function normalizeLevel(string $level): string
    {
        $level = strtolower($level);
        $allowed = ['emergency', 'alert', 'critical', 'error', 'warning', 'notice', 'info', 'debug'];

        if (!in_array($level, $allowed, true)) {
            return 'debug';
        }

        return $level;
    }
}

